/*
 * cIPEndpoint.cpp
 *
 * An IP endpoint.
 */
#include "cIPEndpoint.h"
#include "Endpoint/cEndpointFactory.h"
#include "Util/CoolHash.h"
#include <memory.h>

const unsigned int cIPEndpoint::DEFAULT_SUBNET_MASK = 0xFFFFFF00;

cIPEndpoint::cIPEndpoint()
{
	cEndpoint::mType = ENDPOINT_TYPE_IP;
	memset((void *)&mAddress, 0, sizeof(sockaddr_in));
	mAddress.sin_family = AF_INET;
	mSubnetMask = htonl(cIPEndpoint::DEFAULT_SUBNET_MASK);
	mHashCode = 0;
	mRefCount = 0;
}

cIPEndpoint::cIPEndpoint(char* address, u_short port)
{
	mSubnetMask = htonl(cIPEndpoint::DEFAULT_SUBNET_MASK);
	cEndpoint::mType = ENDPOINT_TYPE_IP;
	memset((void *)&mAddress, 0, sizeof(sockaddr_in));
	mAddress.sin_family = AF_INET;
	mAddress.sin_addr.s_addr = _GetAddr(address);
	mAddress.sin_port = htons(port);
	mRefCount = 0;
	_ComputeHashCode();
}


unsigned int cIPEndpoint::HashCode()
{
	if(!mHashCode)	// re-compute if hashcode is 0
	{
		_ComputeHashCode();
	}
	return mHashCode;
}

unsigned int cIPEndpoint::_ComputeHashCode()
{
	mHashCode = CoolHash((unsigned char *)&mAddress.sin_port, sizeof(mAddress.sin_port)+sizeof(mAddress.sin_addr));
	return mHashCode;
}

/*
 * cIPEndpoint::Print()
 *
 * Purpose:	Prints out the endpoint to the given stream (for debug purposes)
 * IN:		errStream	-> The stream to use for printing.
 * OUT:		-
 * Cond:	-
 * PostCnd:	-
 * Return:	-
 */
void cIPEndpoint::Print(ostream &errStream)
{
	errStream << "@@@@@@@@@@@@@@@@@@@" << endl;
	errStream << "IP:   " << inet_ntoa(mAddress.sin_addr) << endl;
	errStream << "Port: " << ntohs(mAddress.sin_port) << endl;
	if(mAddress.sin_family == AF_INET)
	{
		errStream << "Fam:  " << "AF_INET" << endl;
	}
	else if(mAddress.sin_family == PF_INET)
	{
		errStream << "Fam:  " << "PF_INET" << endl;
	}
	else
	{
		errStream << "Fam:  " << "{UNKNOWN}" << endl;
	}
	errStream << "Hash: " << HashCode() << endl;
	errStream << "This: " << this << endl;
	errStream << "RefC: " << mRefCount << endl;
	errStream << "@@@@@@@@@@@@@@@@@@@" << endl;
}


/*
 * cIPEndpoint::Serialize()
 *
 * Purpose:	Turns the endpoint into a byte stream from in the given buffer.
 * IN:		buffer		-> The buffer in which to place serialized endpoint.
 *			size		-> The size of the buffer provided.
 * OUT:		size		-> The size remaining in the buffer, or the size needed if fail
 * Cond:	-
 * PostCnd:	The endpoint is in the buffer.
 * Return:	The address of the location in the buffer after the serialized representation
 *			OR NULL if failure.
 */
char* cIPEndpoint::Serialize(char* buffer, int *size)
{
	int sizeNeeded = GetSize();
	if(*size < sizeNeeded)
	{
		*size = sizeNeeded;
		return NULL;
	}
	memcpy(buffer, &mType, sizeof(mType));
	buffer += sizeof(mType);
	memcpy(buffer, &mAddress.sin_port, sizeof(mAddress.sin_port)+sizeof(mAddress.sin_addr)); // Address elements already in network order.
	buffer += (sizeof(mAddress.sin_port) + sizeof(mAddress.sin_addr));
	memcpy(buffer, &mSubnetMask, sizeof(mSubnetMask)); // SubnetMask already in network order.
	buffer += sizeof(mSubnetMask);
	*size = *size - sizeNeeded;
	return buffer;
}

/*
 * cIPEndpoint::Deserialize()
 *
 * Purpose:	Fills in the endpoint information from the given buffer.
 * IN:		buffer		-> The buffer in which the endpoint info resides.
 *			size		-> The size of the buffer provided.
 * OUT:		size		-> The size remaining in the buffer, or the size needed if fail
 * Cond:	A valid endpoint of this type is in the buffer.
 * PostCnd:	This endpoint is set up.
 * Return:	The address of the location in the buffer after the serialized representation
 *			OR NULL if failure.
 */
char* cIPEndpoint::Deserialize(char *buffer, int *size)
{
	int sizeNeeded = GetSize();
	if(*size < sizeNeeded)
	{
		*size = sizeNeeded;
		return NULL;
	}
	memcpy(&mType, buffer, sizeof(mType));
	buffer += sizeof(mType);
	memcpy(&mAddress.sin_port, buffer, sizeof(mAddress.sin_port)+sizeof(mAddress.sin_addr));
	buffer += (sizeof(mAddress.sin_port)+sizeof(mAddress.sin_addr));
	memcpy(&mSubnetMask, buffer, sizeof(mSubnetMask));
	buffer += sizeof(mSubnetMask);
	*size = *size - sizeNeeded;

	_ComputeHashCode();

	return buffer;
}

unsigned int cIPEndpoint::GetSize()
{
	return sizeof(mAddress.sin_port) +sizeof(mAddress.sin_addr) + sizeof(mSubnetMask) + sizeof(mType);
}

bool cIPEndpoint::IsSameSubnet(cEndpoint* ep) 
{
	cIPEndpoint* iep;
	if(ep->GetType() == ENDPOINT_TYPE_IP)
	{
		iep = (cIPEndpoint *)ep;
		if(mSubnetMask == iep->mSubnetMask)
		{
			if( (mAddress.sin_addr.s_addr & mSubnetMask) == 
				(iep->mAddress.sin_addr.s_addr & mSubnetMask) )
			{
				return true;
			}
		}
	}
	return false;
}

bool cIPEndpoint::Equals(cHashObject* hObj)
{
	cIPEndpoint* ep = (cIPEndpoint *)hObj;
	if(ep->GetType() != ENDPOINT_TYPE_IP)
	{
		return false;
	}
	else if(ep->HashCode() != HashCode())
	{
		return false;
	}
	else
	{
		return ( (mAddress.sin_port == ep->mAddress.sin_port) && (mAddress.sin_addr.s_addr == ep->mAddress.sin_addr.s_addr) );
	}
}

bool cIPEndpoint::SubnetLessThan(cEndpoint* ep)
{
	if(ep->GetType() != ENDPOINT_TYPE_IP)
	{
		return false;
	}

	cIPEndpoint* iep = (cIPEndpoint *)ep;
	return ((mAddress.sin_addr.s_addr & mSubnetMask) < (iep->mAddress.sin_addr.s_addr & mSubnetMask));
}

bool cIPEndpoint::LessThan(cHashObject* hObj)
{
	cIPEndpoint* ep = (cIPEndpoint *)hObj;
	if(HashCode() < ep->HashCode())
	{
		return true;
	}
	else if(HashCode() > ep->HashCode())
	{
		return false;
	}
	else
	{
		if(mAddress.sin_addr.s_addr < ep->mAddress.sin_addr.s_addr)
		{
			return true;
		}
		else if(mAddress.sin_addr.s_addr > ep->mAddress.sin_addr.s_addr)
		{
			return false;
		}
		else
		{
			if(mAddress.sin_port < ep->mAddress.sin_port)
			{
				return true;
			}
			else
			{
				return false;	// Either it is equal, or greater than!
			}
		}
	}
}

cEndpoint*	cIPEndpoint::AllocCopy()
{
	cIPEndpoint* retVal;
	retVal = (cIPEndpoint *)cEndpointFactory::AllocEndpoint(mType);
	if(retVal)
	{
		*retVal = *this;	// Shallow copy
	}
	return retVal;
}


u_long cIPEndpoint::_GetAddr (char* szHost) 
{
  LPHOSTENT lpstHost;
  u_long lAddr = INADDR_ANY;
  
  /* check that we have a string */
  if (szHost) {
  
    /* check for a dotted-IP address string */
    lAddr = inet_addr (szHost);
  
    /* If not an address, then try to resolve it as a hostname */
    if ((lAddr == INADDR_NONE) && (strcmp (szHost, "255.255.255.255"))) 
	{
      lpstHost = gethostbyname(szHost);
      if (lpstHost) 
	  {  
		/* success */
        lAddr = *((u_long FAR *) (lpstHost->h_addr));
      } 
	  else 
	  {  
        lAddr = INADDR_ANY;   /* failure */
      }
    }
  }
  return (lAddr); 
} /* end GetAddr() */



